/* $Id: filecache.c 25 2006-04-02 14:03:08Z lennart $ */

/***
  This file is part of fusedav.

  fusedav is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  fusedav is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
  License for more details.
  
  You should have received a copy of the GNU General Public License
  along with fusedav; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
***/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <string.h>
#include <limits.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <pthread.h>
#include <inttypes.h>
#include <limits.h>
#include <ctype.h>

#include <ne_props.h>
#include <ne_uri.h>
#include <ne_session.h>
#include <ne_utils.h>
#include <ne_socket.h>
#include <ne_auth.h>
#include <ne_dates.h>
#include <ne_basic.h>
#include <ne_request.h>
#include <ne_locks.h>

#include <curl/curl.h>

#include "filecache.h"
#include "statcache.h"
#include "fusedav.h"
#include "session.h"

#define URI_LEN 1052

struct file_info {
    char *filename;
    int fd;
    CURL *curl;
    off_t server_length, length, present;
    
    int readable;
    int writable;

    int modified;

    int ref, holder, dead;

    pthread_mutex_t mutex;

    /* This field is locked by files_mutex, not by file_info->mutex */
    struct file_info *next;
};

static struct file_info *files = NULL;
static pthread_mutex_t files_mutex = PTHREAD_MUTEX_INITIALIZER;

static int file_cache_sync_unlocked(struct file_info *fi);

int extract_ne_respcode(const char *str) {
    int i = 0;
    char buf[4];

    if (str == NULL || isdigit(str[0]) == 0) {
        return -1;
    }

    for (i = 0; i < 3; i++) {
        buf[i] = str[i];
    }
    buf[3] = '\0';

    return atoi(buf);
}

/*
 * This function calls ne_put inside with 100-Continue expected.
 * Function returns by
 * - -1: if internal error occurs.
 * -  0: if operation finishes successfully.
 * - other negative number: returns status code if http status code indicates error.
 */
int my_ne_put(ne_session *sess, const char *luri, int fd) {
    ne_request *req;
    struct stat64 st;
    int ret;

    if (fstat64(fd, &st)) {
        return NE_ERROR;
    }
    
    req = ne_request_create(sess, "PUT", luri);

    ne_lock_using_resource(req, luri, 0);
    ne_lock_using_parent(req, luri);

    ne_set_request_flag(req, NE_REQFLAG_EXPECT100, 1);
    ne_set_request_body_fd(req, fd, 0, st.st_size);
	
    ret = ne_request_dispatch(req);

    if (ret == NE_OK && ne_get_status(req)->klass != 2)
        ret = NE_ERROR;

    ne_request_destroy(req);

    return ret;
}


void* file_cache_get(const char *path, int holdCnt) {
    struct file_info *f, *r = NULL;

    pthread_mutex_lock(&files_mutex);
    
    for (f = files; f; f = f->next) {
        
        pthread_mutex_lock(&f->mutex);
        if (!f->dead && (f->holder >= 1) && f->filename && !strcmp(path, f->filename)) {
            f->holder += holdCnt;
            if ((holdCnt != -1) || (f->holder == 0)) {
                f->ref++;
                r = f;
	    }
        }
        pthread_mutex_unlock(&f->mutex);

        if (r)
            break;
    }
    
    pthread_mutex_unlock(&files_mutex);
    return f;
}

static void file_cache_free_unlocked(struct file_info *fi) {
    assert(fi && fi->dead && fi->ref == 0);

    free(fi->filename);

    if (fi->fd >= 0)
        close(fi->fd);

    pthread_mutex_destroy(&fi->mutex);
    free(fi);
}

void file_cache_unref(void *f) {
    struct file_info *fi = f;
    assert(fi);

    pthread_mutex_lock(&fi->mutex);

    assert(fi->ref >= 1);
    fi->ref--;

    if (!fi->ref && fi->dead) {
        file_cache_sync_unlocked(fi);
        file_cache_free_unlocked(fi);
    }

    pthread_mutex_unlock(&fi->mutex);
}

static void file_cache_unlink(struct file_info *fi) {
    struct file_info *s, *prev;
    assert(fi);

    pthread_mutex_lock(&files_mutex);
    
    for (s = files, prev = NULL; s; s = s->next) {
        if (s == fi) {
            if (prev)
                prev->next = s->next;
            else
                files = s->next;

            break;
        }
        
        prev = s;
    }
    
    pthread_mutex_unlock(&files_mutex);
}

int file_cache_close(void *f) {
    struct file_info *fi = f;
    int r = 0;
    assert(fi);

    file_cache_unlink(f);

    if (fi->curl != NULL)
        curl_easy_cleanup(fi->curl);

    pthread_mutex_lock(&fi->mutex);
    fi->dead = 1;
    pthread_mutex_unlock(&fi->mutex);

    return r;
}

void* file_cache_open(const char *path, int flags) {
    struct file_info *fi = NULL;
    char tempfile[PATH_MAX];
    const char *length = NULL;
    ne_request *req = NULL;
    ne_session *session;

    if (!(session = session_get(1))) {
        errno = EIO;
        goto fail;
    }

    if ((fi = file_cache_get(path, 1))) {
        if (flags & O_RDONLY || flags & O_RDWR) fi->readable = 1;
        if (flags & O_WRONLY || flags & O_RDWR) fi->writable = 1;
        return fi;
    }

    fi = malloc(sizeof(struct file_info));
    memset(fi, 0, sizeof(struct file_info));
    fi->fd = -1;

    fi->filename = strdup(path);

    if (flags & O_WRONLY || flags & O_RDWR) {
        snprintf(tempfile, sizeof(tempfile), "%s/fusedav-cache-XXXXXX", "/tmp");
        if ((fi->fd = mkstemp(tempfile)) < 0)
            goto fail;
        unlink(tempfile);
    }

    req = ne_request_create(session, "HEAD", path);
    assert(req);

    if (ne_request_dispatch(req) != NE_OK) {
        fprintf(stderr, "HEAD failed: %s\n", ne_get_error(session));
        errno = ENOENT;
        goto fail;
    } else {
        const ne_status *status;
        if ((status = ne_get_status(req)) != NULL) {
            if (status->klass != 2) {
                errno = ENOENT;
                goto fail;
            }
        }
    }

    if (!(length = ne_get_response_header(req, "Content-Length")))
        /* dirty hack, since Apache doesn't send the file size if the file is empty */
        fi->server_length = fi->length = 0; 
    else
        fi->server_length = fi->length = atoi(length);

    ne_request_destroy(req);
    
    if (flags & O_RDONLY || flags & O_RDWR) fi->readable = 1;
    if (flags & O_WRONLY || flags & O_RDWR) fi->writable = 1;

    pthread_mutex_init(&fi->mutex, NULL);
    
    pthread_mutex_lock(&files_mutex);
    fi->next = files;
    files = fi;
    pthread_mutex_unlock(&files_mutex);

    fi->ref = 1;
    fi->holder = 1;

    return fi;

fail:

    if (req)
        ne_request_destroy(req);

    if (fi) {
        if (fi->fd >= 0)
            close(fi->fd);
        free(fi->filename);
        free(fi);
    }
        
    return NULL;
}

static int load_up_to_unlocked(struct file_info *fi, off_t l) {
#ifndef ne_get_range64
#define NE_GET_RANGE ne_get_range
    ne_content_range range;
#else
#define NE_GET_RANGE ne_get_range64
    ne_content_range64 range;
#endif
    ne_session *session;

    assert(fi);

    if (!(session = session_get(1))) {
        errno = EIO;
        return -1;
    }

    if (l > fi->server_length)
        l = fi->server_length;
    
    if (l <= fi->present)
        return 0;

    if (lseek(fi->fd, fi->present, SEEK_SET) != fi->present)
        return -1;
    
    range.start = fi->present;
    range.end = l-1;
    range.total = 0;
    
    if (NE_GET_RANGE(session, fi->filename, &range, fi->fd) != NE_OK) {
        fprintf(stderr, "GET failed: %s\n", ne_get_error(session));
        errno = ENOENT;
        return -1;
    }

    fi->present = l;
    return 0;
#undef NE_GET_RANGE
}

struct wbuf_info {
    char *buf;
    size_t index;
};

static size_t write_data(void *buf, size_t size, size_t nmemb, void *w) {
    struct wbuf_info *winfo = (struct wbuf_info *)w;
    size_t segsize = size * nmemb;

    memcpy(&winfo->buf[winfo->index], buf, segsize);;
    winfo->index += segsize;

    return segsize;
}

int file_cache_read(void *f, char *buf, size_t size, off_t offset) {
    struct file_info *fi = f;
    ssize_t r = -1;
    
    assert(fi && buf && size);

    pthread_mutex_lock(&fi->mutex);

    if (fi->writable) {
        if (load_up_to_unlocked(fi, offset+size) < 0)
            goto finish;

        if ((r = pread(fi->fd, buf, size, offset)) < 0)
            goto finish;

    } else {
        CURLcode ret;
        char range_strings[48];
        struct wbuf_info winfo;
        char uri_strings[URI_LEN];
        size_t range_s = offset;
        size_t range_e = offset + size;
	long rc;

        if (range_e >= fi->server_length)
            range_e = fi->server_length;

        if (range_s < range_e) {
            sprintf(range_strings, "%lld-%lld", (int64_t)range_s, (int64_t)range_e - 1);
            sprintf(uri_strings, "%s://%s:%d%s", uri.scheme ? uri.scheme : "http", uri.host, uri.port ? uri.port : 80, fi->filename);
            winfo.buf = buf;
            winfo.index = 0;

            if (fi->curl == NULL)
                fi->curl = curl_easy_init();

            curl_easy_setopt(fi->curl, CURLOPT_URL, uri_strings);
            curl_easy_setopt(fi->curl, CURLOPT_WRITEDATA, &winfo);
            curl_easy_setopt(fi->curl, CURLOPT_WRITEFUNCTION, write_data);
            curl_easy_setopt(fi->curl, CURLOPT_RANGE, range_strings);

            ret = curl_easy_perform(fi->curl);
            if (ret == 0) {
                ret = curl_easy_getinfo(fi->curl, CURLINFO_RESPONSE_CODE, &rc);
                if (ret == 0 && (rc / 100) == 2) {
                    r = range_e - range_s;
		} else {
                    errno = EIO;
		}
            } else {
                errno = EIO;
	    }
        } else {
            r = 0;
        }
    }

finish:
    
    pthread_mutex_unlock(&fi->mutex);

    return r;
}

int file_cache_write(void *f, const char *buf, size_t size, off_t offset) {
    struct file_info *fi = f;
    ssize_t r = -1;

    assert (fi);

    pthread_mutex_lock(&fi->mutex);

    if (!fi->writable) {
        errno = EBADF;
        goto finish;
    }

    if (load_up_to_unlocked(fi, offset) < 0)
        goto finish;
        
    if ((r = pwrite(fi->fd, buf, size, offset)) < 0)
        goto finish;

    if (offset+size > fi->present)
        fi->present = offset+size;

    if (offset+size > fi->length)
        fi->length = offset+size;

    fi->modified = 1;

finish:
    pthread_mutex_unlock(&fi->mutex);
    
    return r;
}

int file_cache_truncate(void *f, off_t s) {
    struct file_info *fi = f;
    int r;

    assert(fi);

    pthread_mutex_lock(&fi->mutex);

    fi->length = s;
    r = ftruncate(fi->fd, fi->length);

    pthread_mutex_unlock(&fi->mutex);

    return r;
}

int file_cache_sync_unlocked(struct file_info *fi) {
    int r = -1;
    ne_session *session;

    assert(fi);
    
    if (!fi->writable) {
        errno = EBADF;
        goto finish;
    }

    if (!fi->modified) {
        r = 0;
        goto finish;
    }
    
    if (load_up_to_unlocked(fi, (off_t) -1) < 0)
        goto finish;

    if (lseek(fi->fd, 0, SEEK_SET) == (off_t)-1)
        goto finish;

    if (!(session = session_get(1))) {
        errno = EIO;
        goto finish;
    }
    
    if (my_ne_put(session, fi->filename, fi->fd)) {
        const char *err = ne_get_error(session);
        int code = extract_ne_respcode(err);

        fprintf(stderr, "PUT failed: %s\n", err);

        switch (code) {
            case 404:
            case 507:
            default:
                errno = EIO;
                break;
	}

        goto finish;
    }

    fi->modified = 0;
    stat_cache_invalidate(fi->filename);
    dir_cache_invalidate_parent(fi->filename);

    r = 0;

finish:
    
    return r;
}

int file_cache_sync(void *f) {
    struct file_info *fi = f;
    int r = -1;
    assert(fi);

    pthread_mutex_lock(&fi->mutex);
    r = file_cache_sync_unlocked(fi);
    pthread_mutex_unlock(&fi->mutex);
    
    return r;
}

int file_cache_close_all(void) {
    int r = 0;

    pthread_mutex_lock(&files_mutex);

    while (files) {
        struct file_info *fi = files;
        
        pthread_mutex_lock(&fi->mutex);
        fi->ref++;
        pthread_mutex_unlock(&fi->mutex);

        pthread_mutex_unlock(&files_mutex);
        file_cache_close(fi);
        file_cache_unref(fi);
        pthread_mutex_lock(&files_mutex);
    }

    pthread_mutex_unlock(&files_mutex);

    return r;
}

off_t file_cache_get_size(void *f) {
    struct file_info *fi = f;

    assert(fi);

    return fi->length;
}
