/*
 * Copyright (C) 2008 Alp Toker <alp@atoker.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#include "config.h"
#include "FontCustomPlatformData.h"

#include "SharedBuffer.h"
#include "FontPlatformData.h"

#include "WOFFFileFormat.h"

#include <ft2build.h>
#include FT_FREETYPE_H
#include <fontconfig/fontconfig.h>
#include <fontconfig/fcfreetype.h>

#include <QFile>
#include <QTemporaryFile>

#include <wtf/text/CString.h>

namespace WebCore {

FontCustomPlatformData::~FontCustomPlatformData()
{
    if( !m_filename.isEmpty() ) {
        QString tmpfileName = m_filename;
        QFile tmp(tmpfileName);
        tmp.close();
        tmp.remove();
    }
    FcConfigAppFontClear(0);
}

FontPlatformData FontCustomPlatformData::fontPlatformData(int size, bool bold, bool italic, FontOrientation, TextOrientation, FontWidthVariant, FontRenderingMode)
{
	// TODO: Font Face
    //return FontPlatformData(/* m_fontFace, */ size, bold, italic);
    m_fontDescription.setComputedSize(size);
    m_fontDescription.setSpecifiedSize(size);
    m_fontDescription.setItalic(italic);
    m_fontDescription.setWeight(bold ? FontWeightBold : FontWeightNormal);
    return FontPlatformData(m_fontDescription, m_fontDescription.family().family());
}

#if 0
static void releaseData(void* data)
{
    static_cast<SharedBuffer*>(data)->deref();
}
#endif

static FcPattern *queryFont(const FcChar8 *file, const QByteArray &data, int id, FcBlanks *blanks, int *count)
{
    if (data.isEmpty())
        return FcFreeTypeQuery(file, id, blanks, count);

    static FT_Library library = 0;
    if (!library && FT_Init_FreeType(&library)) {
        library = 0;
        return 0;
    }

    FcPattern *pattern = 0;

    FT_Face face;
    if (!FT_New_Memory_Face(library, (const FT_Byte *)data.constData(), data.size(), id, &face)) {
        *count = face->num_faces;

        pattern = FcFreeTypeQueryFace(face, file, id, blanks);

        FT_Done_Face(face);
    }

    return pattern;
}

static void registerFont(const QByteArray &fontData, String *tmpfilename, FontFamily &fontFamily)
{
    FcConfig *config = FcConfigGetCurrent();
    if (!config)
        return;

    FcFontSet *set = FcConfigGetFonts(config, FcSetApplication);
    if (!set) {
        FcConfigAppFontAddFile(config, (const FcChar8 *)":/non-existent");
        set = FcConfigGetFonts(config, FcSetApplication); // try again
        if (!set)
            return;
    }

    QString fileNameForQuery = "";
    QTemporaryFile tmp;

    if (!fontData.isEmpty()) {
        if (!tmp.open())
            return;
        tmp.write(fontData);
        tmp.flush();
        fileNameForQuery = tmp.fileName();
        *tmpfilename = fileNameForQuery;
    }
    tmp.setAutoRemove(false);

    int id = 0;
    FcBlanks *blanks = FcConfigGetBlanks(0);
    int count = 0;

    QString systemLang;

    FcPattern *pattern1 = FcPatternCreate();
    FcDefaultSubstitute(pattern1);
    FcChar8 *lang = 0;
    if (FcPatternGetString(pattern1, FC_LANG, 0, &lang) == FcResultMatch)
        systemLang = QString::fromUtf8((const char *) lang);
    FcPatternDestroy(pattern1);


    FcPattern *pattern = 0;

    do {
        pattern = queryFont((const FcChar8 *)QFile::encodeName(fileNameForQuery).constData(),
                            fontData, id, blanks, &count);
        if (!pattern)
            return;

        FcPatternDel(pattern, FC_FILE);
        QByteArray cs = fileNameForQuery.toUtf8();
        FcPatternAddString(pattern, FC_FILE, (const FcChar8 *) cs.constData());

        FcChar8 *fam = 0, *familylang = 0;
        int i, n = 0;
        for (i = 0; ; i++) {
            if (FcPatternGetString(pattern, FC_FAMILYLANG, i, &familylang) != FcResultMatch)
                break;
            QString familyLang = QString::fromUtf8((const char *) familylang);
            if (familyLang.compare(systemLang, Qt::CaseInsensitive) == 0) {
                n = i;
                break;
            }
        }

        if (FcPatternGetString(pattern, FC_FAMILY, n, &fam) == FcResultMatch) {
            FcPatternDel(pattern, FC_FAMILY);
            static int num = 0;
            char familyalias[1024];
            sprintf(familyalias,"%s%d", (const char *)fam, num);
            fontFamily.setFamily(familyalias);
            FcPatternAddString(pattern, FC_FAMILY, (const FcChar8 *)(familyalias));
            num++;
        }

        if (!FcFontSetAdd(set, pattern))
            return;

        ++id;
    } while (pattern && id < count);
}


FontCustomPlatformData* createFontCustomPlatformData(SharedBuffer* buffer)
{
    // FIXME: we need support in pango to read fonts from memory to implement this.y
    RefPtr<SharedBuffer> sfntBuffer;
    if (isWOFF(buffer)) {
        Vector<char> sfnt;
        if (!convertWOFFToSfnt(buffer, sfnt))
            return 0;

        sfntBuffer = SharedBuffer::adoptVector(sfnt);
        buffer = sfntBuffer.get();
    }

    String filename;
    FontFamily fontFamily;
    registerFont(QByteArray(buffer->data(), buffer->size()), &filename, fontFamily);  

    FontDescription fontDescription;
    fontDescription.setFamily(fontFamily);
    FontCustomPlatformData *data = new FontCustomPlatformData(filename, fontDescription);
    return data;
}

bool FontCustomPlatformData::supportsFormat(const String& format)
{
    return equalIgnoringCase(format, "truetype") || equalIgnoringCase(format, "opentype")
            || equalIgnoringCase(format, "woff");
}

}
