/*
 * Copyright (C) 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2006 Michael Emmel mike.emmel@gmail.com
 * Copyright (c) 2007 Hiroyuki Ikezoe
 * Copyright (c) 2007 Kouhei Sutou
 * Copyright (C) 2007 Alp Toker <alp@atoker.com>
 * Copyright (C) 2008 Xan Lopez <xan@gnome.org>
 * Copyright (C) 2008 Nuanti Ltd.
 * Copyright (C) 2011 ProFUSION embedded systems
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "Font.h"

/* uncomment when you want to dump internal data controlled by getenv. */
/* #define ENABLE_DUMP_BY_GETENV */

/* uncommonet when you want to print performance of Font::drawCommonText2() */
/* #define ENABLE_PERFORMANCE_MEASUREMENT */
#ifdef ENABLE_PERFORMANCE_MEASUREMENT
/* FOR PERF ONLY */
#include <sys/time.h>

static struct timeval tv_s;
static struct timeval tv_e;
static int gs_total = 0;

#define PERF_S() do{ gettimeofday(&tv_s, NULL); }while(0)
#define PERF_E() do{ \
	gettimeofday(&tv_e, NULL); \
	int current = (tv_e.tv_sec - tv_s.tv_sec) * 1000000 + (tv_e.tv_usec - tv_s.tv_usec); \
	gs_total += current; \
	printf(" *** TOOK %d\n", current);\
	printf(" *** TOTAL TO DRAW: %lf\n", gs_total / 1000000.0); \
}while(0)

#else
#define PERF_S()
#define PERF_E()
#endif

//#include "CairoUtilities.h"
#include "GOwnPtr.h"
#include "GraphicsContext.h"
#include "NotImplemented.h"
#include "PlatformContextCairo.h"
#include "ShadowBlur.h"
#include "SimpleFontData.h"
#include "TextRun.h"
#include <pango/pango.h>
//#include <pango/pangocairo.h>

#if PLATFORM(GTK)
#include <gdk/gdk.h>
#else
#include "PangoUtilities.h"
#endif

#if USE(FREETYPE)
#include <pango/pangofc-fontmap.h>
#endif

#include <pango/pangoft2.h>
#include <pango/pango-attributes.h>

#include <QImage>
#include <QPainter>
#include <QPainterPath>

#include <wtf/text/CString.h>

namespace WebCore {

void Font::platformInit() {
	initializePango();
}

void Font::platformDestroy() {
	// 20151109FJ
	// this instance doesn't need to call finalizePango()
	// because finalizePango() destroys global instance.
	// Call finalizePango() when webkit is destroyed.
}

#ifdef GTK_API_VERSION_2
typedef GdkRegion* PangoRegionType;

void destroyPangoRegion(PangoRegionType region)
{
    gdk_region_destroy(region);
}

IntRect getPangoRegionExtents(PangoRegionType region)
{
    GdkRectangle rectangle;
    gdk_region_get_clipbox(region, &rectangle);
    return IntRect(rectangle.x, rectangle.y, rectangle.width, rectangle.height);
}
#endif

#define IS_HIGH_SURROGATE(u)  ((UChar)(u) >= (UChar)0xd800 && (UChar)(u) <= (UChar)0xdbff)
#define IS_LOW_SURROGATE(u)  ((UChar)(u) >= (UChar)0xdc00 && (UChar)(u) <= (UChar)0xdfff)

gchar* utf16ToUtf8(const UChar* aText, gint aLength, gint &length)
{
    gboolean needCopy = FALSE;

    for (int i = 0; i < aLength; i++) {
        if (!aText[i] || IS_LOW_SURROGATE(aText[i])) {
            needCopy = TRUE;
            break;
        } 

        if (IS_HIGH_SURROGATE(aText[i])) {
            if (i < aLength - 1 && IS_LOW_SURROGATE(aText[i+1]))
                i++;
            else {
                needCopy = TRUE;
                break;
            }
        }
    }

    GOwnPtr<UChar> copiedString;
    if (needCopy) {
        /* Pango doesn't correctly handle nuls.  We convert them to 0xff. */
        /* Also "validate" UTF-16 text to make sure conversion doesn't fail. */

        copiedString.set(static_cast<UChar*>(g_memdup(aText, aLength * sizeof(aText[0]))));
        UChar* p = copiedString.get();

        /* don't need to reset i */
        for (int i = 0; i < aLength; i++) {
            if (!p[i] || IS_LOW_SURROGATE(p[i]))
                p[i] = 0xFFFD;
            else if (IS_HIGH_SURROGATE(p[i])) {
                if (i < aLength - 1 && IS_LOW_SURROGATE(aText[i+1]))
                    i++;
                else
                    p[i] = 0xFFFD;
            }
        }

        aText = p;
    }

    gchar* utf8Text;
    glong itemsWritten;
    utf8Text = g_utf16_to_utf8(reinterpret_cast<const gunichar2*>(aText), aLength, 0, &itemsWritten, 0);
    length = itemsWritten;

    return utf8Text;
}

gchar* convertUniCharToUTF8(const UChar* characters, gint length, int from, int to)
{
    gint newLength = 0;
    GOwnPtr<gchar> utf8Text(utf16ToUtf8(characters, length, newLength));
    if (!utf8Text)
        return 0;

    gchar* pos = utf8Text.get();
    if (from > 0) {
        // discard the first 'from' characters
        // FIXME: we should do this before the conversion probably
        pos = g_utf8_offset_to_pointer(utf8Text.get(), from);
    }

    gint len = strlen(pos);
    GString* ret = g_string_new_len(NULL, len);

    // replace line break by space
    while (len > 0) {
        gint index, start;
        pango_find_paragraph_boundary(pos, len, &index, &start);
        g_string_append_len(ret, pos, index);
        if (index == start)
            break;
        g_string_append_c(ret, ' ');
        pos += start;
        len -= start;
    }
    return g_string_free(ret, FALSE);
}

static void setPangoAttributesInternal(const Font* font, PangoLayout* layout);

static void setPangoAttributes(const Font* font, const TextRun& run, PangoLayout* layout)
{
	setPangoAttributesInternal(font, layout);

    PangoContext* pangoContext = pango_layout_get_context(layout);
    PangoDirection direction = run.rtl() ? PANGO_DIRECTION_RTL : PANGO_DIRECTION_LTR;
    pango_context_set_base_dir(pangoContext, direction);
    PangoAttrList* list = pango_attr_list_new();
    PangoAttribute* attr;

    attr = pango_attr_size_new_absolute(font->pixelSize() * PANGO_SCALE);
    attr->end_index = G_MAXUINT;
    pango_attr_list_insert_before(list, attr);

    if (!run.spacingDisabled()) {
        attr = pango_attr_letter_spacing_new(font->letterSpacing() * PANGO_SCALE);
        attr->end_index = G_MAXUINT;
        pango_attr_list_insert_before(list, attr);
    }
    pango_layout_set_attributes(layout, list);
    pango_attr_list_unref(list);
}

/* when setPangoAttributes called without textrun, direction and spacing is controlled
   by callers.
 */
static void setPangoAttributesInternal(const Font* font, PangoLayout* layout)
{
    PangoFontDescription* desc = pango_font_description_new();
#if USE(FREETYPE)
    if (font->primaryFont()->platformData().m_pattern) {
        desc = pango_fc_font_description_from_pattern(font->primaryFont()->platformData().m_pattern.get(), FALSE);
        pango_layout_set_font_description(layout, desc);
    }
#elif 1 // USE(PANGO)
    if (font->primaryFont()->platformData().m_font) {
        desc = pango_font_describe(font->primaryFont()->platformData().m_font);
        pango_layout_set_font_description(layout, desc);
    }
#endif

    pango_layout_set_auto_dir(layout, FALSE);
    PangoAttrList* list = pango_attr_list_new();
    PangoAttribute* attr;

    attr = pango_attr_size_new_absolute(font->pixelSize() * PANGO_SCALE);
    attr->end_index = G_MAXUINT;
    pango_attr_list_insert_before(list, attr);

    // Pango does not yet support synthesising small caps
    // See http://bugs.webkit.org/show_bug.cgi?id=15610

    pango_layout_set_attributes(layout, list);
    pango_attr_list_unref(list);

//	int italic_offset = 0;
    /* set font-style*/
	if(font->italic() == FontItalicOn) {
		pango_font_description_set_style(desc, PANGO_STYLE_ITALIC);
//		italic_offset = (int)((font->pixelSize() / tan((90 - 15) * 3.14 / 180)) + 0.5);
	}

	FontWeight webkit_weight = font->weight();
	if(webkit_weight != FontWeightNormal) {
		PangoWeight pango_weight;
		switch(webkit_weight) {
			case FontWeight100:
				pango_weight = PANGO_WEIGHT_THIN /* = 100 */ ;
				break;
			case FontWeight200:
				pango_weight = PANGO_WEIGHT_ULTRALIGHT /* = 200 */ ;
				break;
			case FontWeight300:
				pango_weight = PANGO_WEIGHT_LIGHT /* = 300 */ ;
				break;
			case FontWeight400: /* equal to FontWeightNormal */
				pango_weight = PANGO_WEIGHT_NORMAL /* = 400 */ ;
				break;
			case FontWeight500:
				pango_weight = PANGO_WEIGHT_MEDIUM /* = 500 */;
				break;
			case FontWeight600:
				pango_weight = PANGO_WEIGHT_SEMIBOLD /* = 600 */ ;
				break;
			case FontWeight700: /* equal to FontWeightBold */
				pango_weight = PANGO_WEIGHT_BOLD /* = 700 */ ;
				break;
			case FontWeight800:
				pango_weight = PANGO_WEIGHT_ULTRABOLD /* = 800 */ ;
				break;
			case FontWeight900:
				pango_weight = PANGO_WEIGHT_HEAVY /* = 900 */ ;
				break;
			default:
				pango_weight = PANGO_WEIGHT_NORMAL /* = 400 */ ;
		}
		pango_font_description_set_weight(desc, pango_weight);
	}
    
    pango_layout_set_font_description (layout, desc);
    pango_font_description_free(desc);
}

bool Font::canReturnFallbackFontsForComplexText()
{
    return false;
}

bool Font::canExpandAroundIdeographsInComplexText()
{
    return false;
}

static void pango_glyph_item_word_space (PangoLayout* layout,
			          int          word_spacing)
{
    if (word_spacing == 0)
    {
        return;
    }
    PangoLayoutIter* iter;
    gboolean has_more = true;
    const char* text= pango_layout_get_text(layout);
    QString str = QString::fromUtf8(text);
    iter = pango_layout_get_iter (layout); 
    pango_layout_iter_get_index(iter);
    int before_num_glyphs = 0;
    while (has_more)
    {
        PangoLayoutRun* pangoRun = pango_layout_iter_get_run(iter);
        has_more = pango_layout_iter_next_run(iter);
        if (pangoRun)
        {
        
            PangoGlyphString* glyphString = ((PangoGlyphItem*)pangoRun)->glyphs;
            PangoItem* item = ((PangoGlyphItem*)pangoRun)->item;
            if (glyphString && item)
            {
                int num = glyphString->num_glyphs;
                for (int i = 0; i < num; i++)
                {
                    int pos = before_num_glyphs + i;
                    QChar ch = str.at(pos);
                    if(ch.isSpace())
                    {
                        glyphString->glyphs[i].geometry.width = word_spacing*PANGO_SCALE;
                        //glyphString->glyphs[i].geometry.x_offset += word_spacing*PANGO_SCALE;                    
                    }
                }
                before_num_glyphs += num;
            }
        }
    }
    pango_layout_iter_free (iter);
}

#define GRID(x, y) grid[(y)*(w+1) + (x)]
#define SET(x, y) (*(image_data + (y)*bpl + ((x) >> 3)) & (0x80 >> ((x) & 7)))

enum { EdgeRight = 0x1,
       EdgeDown = 0x2,
       EdgeLeft = 0x4,
       EdgeUp = 0x8
};

static void collectSingleContour(qreal x0, qreal y0, uint *grid, int x, int y, int w, int h, QPainterPath *path)
{
    Q_UNUSED(h);

    path->moveTo(x + x0, y + y0);    
    while (GRID(x, y)) {
        if (GRID(x, y) & EdgeRight) {
            while (GRID(x, y) & EdgeRight) {
                GRID(x, y) &= ~EdgeRight;
                ++x;
            }
            Q_ASSERT(x <= w);
            path->lineTo(x + x0, y + y0);            
            continue;
        }
        if (GRID(x, y) & EdgeDown) {
            while (GRID(x, y) & EdgeDown) {
                GRID(x, y) &= ~EdgeDown;
                ++y;
            }
            Q_ASSERT(y <= h);
            path->lineTo(x + x0, y + y0);            
            continue;
        }
        if (GRID(x, y) & EdgeLeft) {
            while (GRID(x, y) & EdgeLeft) {
                GRID(x, y) &= ~EdgeLeft;
                --x;
            }
            Q_ASSERT(x >= 0);
            path->lineTo(x + x0, y + y0);            
            continue;
        }
        if (GRID(x, y) & EdgeUp) {
            while (GRID(x, y) & EdgeUp) {
                GRID(x, y) &= ~EdgeUp;
                --y;
            }
            Q_ASSERT(y >= 0);
            path->lineTo(x + x0, y + y0);            
            continue;
        }
    }
    path->closeSubpath();
}

void qt_addBitmapToPath(qreal x0, qreal y0, const uchar *image_data, int bpl, int w, int h, QPainterPath *path)
{
    uint *grid = new uint[(w+1)*(h+1)];
    // set up edges
    for (int y = 0; y <= h; ++y) {
        for (int x = 0; x <= w; ++x) {
            bool topLeft = (x == 0)|(y == 0) ? false : SET(x - 1, y - 1);
            bool topRight = (x == w)|(y == 0) ? false : SET(x, y - 1);
            bool bottomLeft = (x == 0)|(y == h) ? false : SET(x - 1, y);
            bool bottomRight = (x == w)|(y == h) ? false : SET(x, y);

            GRID(x, y) = 0;
            if ((!topRight) & bottomRight)
                GRID(x, y) |= EdgeRight;
            if ((!bottomRight) & bottomLeft)
                GRID(x, y) |= EdgeDown;
            if ((!bottomLeft) & topLeft)
                GRID(x, y) |= EdgeLeft;
            if ((!topLeft) & topRight)
                GRID(x, y) |= EdgeUp;            
        }        
    }

    // collect edges
    for (int y = 0; y < h; ++y) {
        for (int x = 0; x < w; ++x) {
            if (!GRID(x, y))
                continue;
            // found start of a contour, follow it
            collectSingleContour(x0, y0, grid, x, y, w, h, path);
        }
    }
    delete [] grid;
}

#undef GRID
#undef SET

static QPen fillPenForContext(GraphicsContext* ctx)
{
    if (ctx->fillGradient()) {
        QBrush brush(*ctx->fillGradient()->platformGradient());
        brush.setTransform(ctx->fillGradient()->gradientSpaceTransform());
        return QPen(brush, 0);
    }

    if (ctx->fillPattern()) {
        AffineTransform affine;
        return QPen(QBrush(ctx->fillPattern()->createPlatformPattern(affine)), 0);
    }

    return QPen(QColor(ctx->fillColor()));
}

static QPen strokePenForContext(GraphicsContext* ctx)
{
    if (ctx->strokeGradient()) {
        QBrush brush(*ctx->strokeGradient()->platformGradient());
        brush.setTransform(ctx->strokeGradient()->gradientSpaceTransform());
        return QPen(brush, ctx->strokeThickness());
    }

    if (ctx->strokePattern()) {
        AffineTransform affine;
        QBrush brush(ctx->strokePattern()->createPlatformPattern(affine));
        return QPen(brush, ctx->strokeThickness());
    }

    return QPen(QColor(ctx->strokeColor()), ctx->strokeThickness());
}


#define RGB8TOPANGOCOLOR(a)	((a) * 65535 / 256)

static void get_text_drawing_properties(
	GraphicsContext* context, const FloatPoint& point, PangoLayout* layout,

	/* [IN] */ short word_spacing,
	/* [IN] : italic() == FontItalicOn */ bool isItalic,
	/* [IN] */ int pixelSize,

	/* [OUT] */ int& width,
	/* [OUT] */ int& height,
	/* [OUT] */ int& xOffset,
	/* [OUT] */ int& yOffset,
	/* [OUT] */ float& tX,
	/* [OUT] */ float& tY,
	/* [OUT] */ int& averageBaseline
)
{
    /* word-spacing */
    pango_glyph_item_word_space(layout, word_spacing);

    PangoRectangle ink_rect;
    PangoRectangle logical_rect;
	pango_layout_get_pixel_extents(layout, &ink_rect, &logical_rect);

	int italic_offset = 0;
    /* set font-style*/
	if(isItalic) {
    	italic_offset = (int)((pixelSize / tan((90 - 15) * 3.14 / 180)) + 0.5);
	}

    width = MAX(logical_rect.x + logical_rect.width, PANGO_PIXELS(pango_layout_get_width(layout))) + italic_offset;
    height = MAX(logical_rect.y + logical_rect.height, PANGO_PIXELS(pango_layout_get_height(layout)));

    xOffset = logical_rect.x;
    yOffset = logical_rect.y;
    tX = point.x();
    averageBaseline = pango_layout_get_baseline (layout)/PANGO_SCALE;
    tY = point.y() - averageBaseline;

    if (ink_rect.x < logical_rect.x)
    {
        xOffset = logical_rect.x - ink_rect.x;
        tX = tX - xOffset;
        width = MAX(width, ink_rect.width);
    }
    else
    {
        width = MAX(width, ink_rect.x + ink_rect.width);
    }

    if (ink_rect.y < logical_rect.y)
    {
        yOffset = logical_rect.y - ink_rect.y;
        tY = tY - yOffset;
        height = MAX(height, ink_rect.height);
    }
    else
    {
        height = MAX(height, ink_rect.y + ink_rect.height);
    }
}

static QImage translateQImageForPath(QImage& image, int width, int height) {
    const int srcBpl = image.bytesPerLine();
    QImage qbitmap = QImage(width, height, QImage::Format_Mono);
    const uchar *imageData = image.bits();
    const int destBpl = qbitmap.bytesPerLine();
    uchar *bitmapData = qbitmap.bits();

    for (int yi = 0; yi < height; ++yi) {
        const uchar *src = imageData + yi*srcBpl;
        uchar *dst = bitmapData + yi*destBpl;
        for (int xi = 0; xi < width; ++xi) {
            const int byte = xi / 8;
            const int bit = xi % 8;
            if (bit == 0)
                dst[byte] = 0;
            if (src[xi])
                dst[byte] |= 128 >> bit;
        }
    }
	return qbitmap;
}

#define USE_FJIB_PARTIAL_DRAW_TEXT

// specify partial image width / height (ex. OpenGL max texture size)
#define FJIB_LARGE_IMAGE_WIDTH_THRESHOLD	(8192)
#define FJIB_LARGE_IMAGE_HEIGHT_THRESHOLD	(8192)

static void drawTextRunImage(QPainter* painter, const QPointF point, QImage& textrun_image) {
#ifdef USE_FJIB_PARTIAL_DRAW_TEXT
	int width = textrun_image.width();
	int height = textrun_image.height();

	QPointF _point = point;
	QImage partialImage;

	if(	width > FJIB_LARGE_IMAGE_WIDTH_THRESHOLD ||
		height > FJIB_LARGE_IMAGE_HEIGHT_THRESHOLD) {

		int cols = width / FJIB_LARGE_IMAGE_WIDTH_THRESHOLD;
		int rows = height / FJIB_LARGE_IMAGE_HEIGHT_THRESHOLD;

		int srcx = 0;
		int srcy = 0;

		int last_width = width % FJIB_LARGE_IMAGE_WIDTH_THRESHOLD;
		int last_height = height % FJIB_LARGE_IMAGE_HEIGHT_THRESHOLD;

		int i;
		int j;
		for(i = 0; i < rows; i++) {
			for(j = 0; j < cols; j++) {
				partialImage = textrun_image.copy(srcx, srcy, FJIB_LARGE_IMAGE_WIDTH_THRESHOLD, FJIB_LARGE_IMAGE_HEIGHT_THRESHOLD);
				painter->drawImage(_point, partialImage);
				srcx += FJIB_LARGE_IMAGE_WIDTH_THRESHOLD;
				_point.rx() = point.x() + srcx;
			}
			// last piece of row
			{
				partialImage = textrun_image.copy(srcx, srcy, last_width, FJIB_LARGE_IMAGE_HEIGHT_THRESHOLD);
				painter->drawImage(_point, partialImage);
			}
			_point.setX(point.x());
			srcy += FJIB_LARGE_IMAGE_HEIGHT_THRESHOLD;
			_point.ry() = point.y() + srcy;
			srcx = 0;
		}
		// last line
		srcx = 0;
		for(j = 0; j < cols; j++) {
			partialImage = textrun_image.copy(srcx, srcy, FJIB_LARGE_IMAGE_WIDTH_THRESHOLD, last_height);
			painter->drawImage(_point, partialImage);
			srcx += FJIB_LARGE_IMAGE_WIDTH_THRESHOLD;
			_point.rx() = point.x() + srcx;
		}
		// last piece of row
		{
			partialImage = textrun_image.copy(srcx, srcy, last_width, last_height);
			painter->drawImage(_point, partialImage);
		}
	}
	else {
		painter->drawImage(point, textrun_image);
	}
#else
	painter->drawImage(point, textrun_image);
#endif
}

static void drawFilledText(GraphicsContext* context, const FloatPoint& target_point, int width, int height, QImage& textrun_image) {
	QBrush brush;

	if(context->fillGradient()) {
        QGradient gd = *context->fillGradient()->platformGradient();
		brush = QBrush(gd);
        brush.setTransform(context->fillGradient()->gradientSpaceTransform());
        if(gd.type() == QGradient::RadialGradient)
            brush.setTransform(QTransform::fromTranslate(-target_point.x(), -target_point.y()));
	}
    else if (context->fillPattern()) {
        AffineTransform affine;
        brush = QBrush(context->fillPattern()->createPlatformPattern(affine));
        brush.setTransform(QTransform::fromTranslate(-target_point.x(), -target_point.y()));
    }
	else {
		qWarning() << " *** drawFilledText() called without fillGradiant or fillPattern ***";
		return;
	}

	QImage reference_image(width, height, QImage::Format_ARGB32);
	QImage output_image(width, height, QImage::Format_ARGB32);
	QPainter painter(&reference_image);
	painter.fillRect(0, 0, width, height, brush);

	uchar* pdst;
	uchar* psrc;
	for(int j = 0; j < height; j++) {
		pdst = output_image.scanLine(j);
		psrc = reference_image.scanLine(j);
		for(int i = 0; i < width; i++) {
			int idx = textrun_image.pixelIndex(i, j);
			(*pdst++) = (*psrc++);		// B
			(*pdst++) = (*psrc++);		// G
			(*pdst++) = (*psrc++);		// R
			(*pdst++) = idx;			// A
			psrc++;
		}
	}
	QPainter* target_painter = context->platformContext();
	drawTextRunImage(target_painter, target_point, output_image);
}

static void drawCommonTextInternal(GraphicsContext* context, QImage& textrun_image,
	int width, int height, const FloatPoint& target_point, bool hasShadow) {

	/* setup painter -s */
    QPainter* painter = context->platformContext();

    painter->setRenderHint(QPainter::SmoothPixmapTransform, true);

    QPen textStrokePen;
    if (context->textDrawingMode() & TextModeStroke)
        textStrokePen = strokePenForContext(context);

    QPainterPath textPath;
    if (context->textDrawingMode() & TextModeStroke) {
		QImage qbitmap = (textrun_image.depth() == 1) ? textrun_image : translateQImageForPath(textrun_image, width, height);
        qt_addBitmapToPath(target_point.x(), target_point.y(), qbitmap.bits(), qbitmap.bytesPerLine(), width, height, &textPath);
	}
	/* setup painter -e */

	Color color;
	if(context->state().textDrawingMode & TextModeFill) {
		color = context->state().fillColor;
	}
	else {
		color = context->state().strokeColor;
	}

	for(int i = 0; i < 256 ; ++i) { 
		textrun_image.setColor(i, qRgba(color.red(), color.green(), color.blue(), i*color.alpha()/255.0f)); 
	}

	if(hasShadow) {
		ShadowBlur* ctxShadow = context->shadowBlur();
		if (context->textDrawingMode() & TextModeFill) 
		{
			Color color = context->state().shadowColor;
			for(int i = 0; i < 256 ; ++i) { 
				textrun_image.setColor(i, qRgba(color.red(), color.green(), color.blue(), i*color.alpha()/255.0f)); 
			}
			if (ctxShadow->type() != ShadowBlur::BlurShadow) 
			{
				QPointF shadowOffset = QPointF(context->state().shadowOffset.width(), context->state().shadowOffset.height());
				shadowOffset += target_point;
				drawTextRunImage(painter, shadowOffset, textrun_image);
			} 
			else 
			{
				QRectF boundingRect(target_point.x(),target_point.y(),width,height);
				GraphicsContext* shadowContext = ctxShadow->beginShadowLayer(context, boundingRect);
				if (shadowContext) 
				{
					QPainter* shadowPainter = shadowContext->platformContext();
					drawTextRunImage(shadowPainter, target_point, textrun_image);
					ctxShadow->endShadowLayer(context);
				}
			}
		}
		else if(context->textDrawingMode() & TextModeStroke) {
			if (ctxShadow->type() != ShadowBlur::BlurShadow) {
				QPointF shadowOffset = QPointF(context->state().shadowOffset.width(), context->state().shadowOffset.height());
				painter->translate(shadowOffset);                
				painter->strokePath(textPath, QPen(context->state().shadowColor));
				painter->translate(-shadowOffset);
			} else {
				QRectF boundingRect(target_point.x(),target_point.y(),width,height);
				GraphicsContext* shadowContext = ctxShadow->beginShadowLayer(context, boundingRect);
				if (shadowContext) {
					QPainter* shadowPainter = shadowContext->platformContext();
					shadowPainter->strokePath(textPath, QPen(context->state().shadowColor));
					ctxShadow->endShadowLayer(context);
				}
			}            
		}
	}
	else {
		if (context->textDrawingMode() & TextModeStroke) {
			const bool antiAlias = painter->testRenderHint(QPainter::HighQualityAntialiasing);
			painter->setRenderHint(QPainter::HighQualityAntialiasing, true);
		    painter->strokePath(textPath, textStrokePen);
			painter->setRenderHint(QPainter::HighQualityAntialiasing, antiAlias);
		}

		if (context->textDrawingMode() & TextModeFill) {
			if(context->fillGradient() || context->fillPattern()) {
				drawFilledText(context, target_point, width, height, textrun_image);
			}
		    else {
				drawTextRunImage(painter, target_point, textrun_image);
			}
		}
	}

#ifdef ENABLE_DUMP_BY_GETENV
	if(getenv("FJIB_DUMP_TEXTRUN")) {
		static int s_count = 0;
		{
			char filename[256];
			sprintf(filename, "./textrun%04d.png", s_count++);
			textrun_image.save(QString::fromUtf8(filename));

			printf(" *** dumped image: %s\n", filename);
		}
	}
#endif
}

void Font::drawCommonText2(GraphicsContext* context, const GlyphBuffer& glyphBuffer, int length,
	int from, const FloatPoint& point, PangoFontMap* map, PangoLayout* layout, bool hasShadow) const {
	int width, height;
	int xOffset, yOffset;
	float tX, tY;
	int averageBaseline;

	PERF_S();

	int to = from + length;

	const UChar* ch = glyphBuffer.glyphs(from);
    gchar* utf8 = convertUniCharToUTF8(ch, length, 0, length);
    pango_layout_set_text(layout, utf8, -1);
#ifdef ENABLE_DUMP_BY_GETENV
	if(getenv("FJIB_DUMP_STRING_TO_DRAW")) {
		printf(" *** UTF-8(%d): %s\n", length, utf8);
	}
#endif

	get_text_drawing_properties(
		context, point, layout,
		/* [IN] */ wordSpacing(),
		/* [IN] */ italic() == FontItalicOn,
		/* [IN] */ pixelSize(),
		/* [OUT] */ width,
		/* [OUT] */ height,
		/* [OUT] */ xOffset,
		/* [OUT] */ yOffset,
		/* [OUT] */ tX,
		/* [OUT] */ tY,
		/* [OUT] */ averageBaseline
	);
	g_free(utf8);

	QPointF target_point(tX, tY);

#if 0
	printf(" *** width = %d, height = %d\n", width, height);
	printf(" *** tX = %f, tY = %f\n", tX, tY);
#endif

	int advance = 0;
	for(int i = from; i < to; i++) {
		advance += glyphBuffer.advanceAt(i);
	}
	ASSERT(advance != 0);

	if(advance > width) {
		width = advance;
	}

	if(italic() == FontItalicOn) {
    	width += (int)((pixelSize() / tan((90 - 15) * 3.14 / 180)) + 0.5);
	}

#if 0
	printf(" *** width + advance(%d) = %d\n", advance, width);
	printf(" *** xOffset = %d, yOffset = %d\n", xOffset, yOffset);
	printf(" *** tX, tY = %f, %f\n", tX, tY);
	printf(" *** point.x() = %f, point.y() = %f\n", point.x(), point.y());

	printf(" *** from: %d, length: %d\n", from, length);
#endif

	int xCurrentOffset = xOffset;
	gchar buffer[16];

	FT_Bitmap *bitmap = pangoft2_view_request(width, height);
	if(bitmap) {
		for(int i = from; i < to; i++) {
#if 0
			utf8 = utf16ToUtf8(ch, 1, converted);
#else
			gint length = g_unichar_to_utf8((gunichar)(*ch), buffer);
			buffer[length] = '\0';
#endif
			pango_layout_set_text(layout, /*utf8*/buffer, -1 /* length */);

#if 1
			int singleBaseline = pango_layout_get_baseline (layout)/PANGO_SCALE;
			pango_ft2_render_layout((FT_Bitmap *)bitmap, layout, xCurrentOffset, yOffset + averageBaseline - singleBaseline);
#else
			pango_ft2_render_layout((FT_Bitmap *)bitmap, layout, xCurrentOffset, yOffset);
#endif

			ch++;
			xCurrentOffset += glyphBuffer.advanceAt(i);
#if 0
			g_free(utf8);
#endif
		}

		QImage textrun_image(bitmap->buffer, width, height, bitmap->pitch, QImage::Format_Indexed8);
		drawCommonTextInternal(context, textrun_image, width, height, target_point, hasShadow);
	}
	else {
		qWarning() << " *** !!! NO BITMAP !!! ***";
	}

	PERF_E();
}

void Font::drawCommonText(GraphicsContext* context, const UChar* ch, int length, const FloatPoint& point, PangoFontMap* map, PangoLayout* layout, bool hasShadow) const
{
	int width, height;
	int xOffset, yOffset;
	float tX, tY;
	int averageBaseline;

    gchar* utf8 = convertUniCharToUTF8(ch, length, 0, length);
    pango_layout_set_text(layout, utf8, -1);

	get_text_drawing_properties(
		context, point, layout,
		/* [IN] */ wordSpacing(),
		/* [IN] */ italic() == FontItalicOn,
		/* [IN] */ pixelSize(),
		/* [OUT] */ width,
		/* [OUT] */ height,
		/* [OUT] */ xOffset,
		/* [OUT] */ yOffset,
		/* [OUT] */ tX,
		/* [OUT] */ tY,
		/* [OUT] */ averageBaseline
	);
	QPointF target_point(tX, tY);

#if 0
	printf(" *** [drawCommonText] UTF-8(%d): %s\n", length, utf8);
	printf(" *** [drawCommonText] width = %d, height = %d\n", width, height);
#endif

    g_free(utf8);

	FT_Bitmap *bitmap = pangoft2_view_request(width, height);
	if(bitmap) {
		pango_ft2_render_layout ((FT_Bitmap *)bitmap,layout, xOffset, yOffset);
		QImage textrun_image(bitmap->buffer, width, height, bitmap->pitch, QImage::Format_Indexed8);
		drawCommonTextInternal(context, textrun_image, width, height, target_point, hasShadow);
	}
}

void Font::drawComplexText(GraphicsContext* context, const TextRun& run, const FloatPoint& point, int from, int to) const
{
#if USE(FREETYPE)
    if (!primaryFont()->platformData().m_pattern) {
        drawSimpleText(context, run, point, from, to);
        return;
    }
#endif

	pango_context_set_base_dir(PANGO_CONTEXT_INSTANCE, PANGO_DIRECTION_LTR);
	pango_context_set_language(PANGO_CONTEXT_INSTANCE, PANGO_LANGUAGE_INSTANCE);

	PangoLayout* layout = pango_layout_new(PANGO_CONTEXT_INSTANCE);
    setPangoAttributes(this, run, layout);

	const UChar* ch = run.characters() + from;
	int length = (from > to) ? from - to : to - from;

	// recalculate target point
	FloatPoint target_point;
	if(run.length() == length) {
		target_point = point;
	}
	else {
		FloatRect fr = selectionRectForComplexText(run, point, pixelSize(), from, to);
		target_point = fr.location();
	}

    drawCommonText(context, ch, length, target_point, PANGO_MAP_INSTANCE, layout);

    g_object_unref(layout);
}

void Font::drawGlyphs(GraphicsContext* context, const SimpleFontData* font,
                      const GlyphBuffer& glyphBuffer,  int from, int numGlyphs,
                      const FloatPoint& point) const
{
	pango_context_set_base_dir(PANGO_CONTEXT_INSTANCE, PANGO_DIRECTION_LTR);
	pango_context_set_language(PANGO_CONTEXT_INSTANCE, PANGO_LANGUAGE_INSTANCE);
	PangoLayout* layout = pango_layout_new(PANGO_CONTEXT_INSTANCE);

    if (font->platformData().m_font) {
        PangoFontDescription* desc = pango_font_describe(font->platformData().m_font);
        pango_layout_set_font_description(layout, desc);
        pango_font_description_free(desc);
    }

    if(context->hasShadow() && context->shadowBlur()->type() != ShadowBlur::NoShadow) {
		drawCommonText2(context, glyphBuffer, numGlyphs, from, point, PANGO_MAP_INSTANCE, layout, true);
    }

	drawCommonText2(context, glyphBuffer, numGlyphs, from, point, PANGO_MAP_INSTANCE, layout);

	g_object_unref(layout);
}

void Font::drawEmphasisMarksForComplexText(GraphicsContext* /* context */, const TextRun& /* run */, const AtomicString& /* mark */, const FloatPoint& /* point */, int /* from */, int /* to */) const
{
    notImplemented();
}

float Font::floatWidthForComplexText(const TextRun& run, HashSet<const SimpleFontData*>* fallbackFonts, GlyphOverflow* overflow) const
{
#if USE(FREETYPE)
    if (!primaryFont()->platformData().m_pattern)
        return floatWidthForSimpleText(run, 0, fallbackFonts, overflow);
#endif

    if (!run.length())
        return 0.0f;

	PangoLayout* layout = pango_layout_new(PANGO_CONTEXT_INSTANCE);
    setPangoAttributes(this, run, layout);

    gchar* utf8 = convertUniCharToUTF8(run.characters(), run.length(), 0, run.length());
    pango_layout_set_text(layout, utf8, -1);

    int width;
#if 0
    pango_layout_get_pixel_size(layout, &width, 0);
#else
    PangoRectangle ink_rect;
    PangoRectangle logical_rect;
    pango_layout_get_pixel_extents(layout, &ink_rect, &logical_rect);

    /* set font-style*/
    width = MAX(logical_rect.x + logical_rect.width, PANGO_PIXELS(pango_layout_get_width(layout)));
#endif

    g_free(utf8);
    g_object_unref(layout);

    // WebKit expects us to ignore word spacing on the first character (as opposed to what Qt does)
    if (treatAsSpace(run[0]))
        width -= m_wordSpacing;
    return width + run.expansion();
}

int Font::offsetForPositionForComplexText(const TextRun& run, float xFloat, bool includePartialGlyphs) const
{
#if USE(FREETYPE)
    if (!primaryFont()->platformData().m_pattern)
        return offsetForPositionForSimpleText(run, xFloat, includePartialGlyphs);
#endif
    // FIXME: This truncation is not a problem for HTML, but only affects SVG, which passes floating-point numbers
    // to Font::offsetForPosition(). Bug http://webkit.org/b/40673 tracks fixing this problem.
    int x = static_cast<int>(xFloat);

    PangoLayout* layout = pango_layout_new(PANGO_CONTEXT_INSTANCE);
    setPangoAttributes(this, run, layout);

    gchar* utf8 = convertUniCharToUTF8(run.characters(), run.length(), 0, run.length());
    pango_layout_set_text(layout, utf8, -1);
    /* word-spacing */
    short word_spacing = wordSpacing();
    pango_glyph_item_word_space(layout, word_spacing);

    int index, trailing;
    pango_layout_xy_to_index(layout, x * PANGO_SCALE, 1, &index, &trailing);
    glong offset = g_utf8_pointer_to_offset(utf8, utf8 + index);
    if (includePartialGlyphs)
        offset += trailing;

    g_free(utf8);
    g_object_unref(layout);

    return offset;
}

FloatRect Font::selectionRectForComplexText(const TextRun& run, const FloatPoint& point, int h, int from, int to) const
{
#if USE(FREETYPE)
    if (!primaryFont()->platformData().m_pattern)
        return selectionRectForSimpleText(run, point, h, from, to);
#endif

    PangoLayout* layout = pango_layout_new(PANGO_CONTEXT_INSTANCE);
    setPangoAttributes(this, run, layout);

    gchar* utf8 = convertUniCharToUTF8(run.characters(), run.length(), 0, run.length());
    pango_layout_set_text(layout, utf8, -1);
    /* word-spacing */
    short word_spacing = wordSpacing();
    pango_glyph_item_word_space(layout, word_spacing);

    char* start = g_utf8_offset_to_pointer(utf8, from);
    char* end = g_utf8_offset_to_pointer(start, to - from);

    if (run.ltr()) {
        from = start - utf8;
        to = end - utf8;
    } else {
        from = end - utf8;
        to = start - utf8;
    }

    PangoLayoutLine* layoutLine = pango_layout_get_line_readonly(layout, 0);
    int xPos;

    xPos = 0;
    if (from < layoutLine->length)
        pango_layout_line_index_to_x(layoutLine, from, FALSE, &xPos);
    float beforeWidth = PANGO_PIXELS_FLOOR(xPos);

    xPos = 0;
    if (run.ltr() || to < layoutLine->length)
        pango_layout_line_index_to_x(layoutLine, to, FALSE, &xPos);
    float afterWidth = PANGO_PIXELS(xPos);

    g_free(utf8);
    g_object_unref(layout);

    return FloatRect(point.x() + beforeWidth, point.y(), afterWidth - beforeWidth, h);
}

}
