/*
 * Copyright (C) 2006 Apple Computer, Inc.
 * Copyright (C) 2006 Michael Emmel mike.emmel@gmail.com
 * Copyright (C) 2007, 2008 Alp Toker <alp@atoker.com>
 * Copyright (C) 2007 Holger Hans Peter Freyther
 * Copyright (C) 2007 Pioneer Research Center USA, Inc.
 * Copyright (C) 2009 Igalia S.L.
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#include <FjLogger.h>

#include "config.h"
#include "FontPlatformData.h"
#include "PangoUtilities.h"
#include <fontconfig/fontconfig.h>
#include <string.h>

#include "PlatformString.h"
#include "FontDescription.h"
#include <wtf/text/CString.h>
//#include <cairo.h>
//#include <assert.h>

#include <pango/pango.h>
//#include <pango/pangocairo.h>

// Use cairo-ft i a recent enough Pango version isn't available
#if !PANGO_VERSION_CHECK(1,18,0)
#include <cairo-ft.h>
#include <pango/pangofc-fontmap.h>
#endif

#include <pango/pangoft2.h>

#include <boost/unordered_map.hpp>
#include <string>


namespace WebCore {

static inline bool isWhitespace(UChar c)
{
    return c == ' ' || c == '\n' || c == '\r' || c == '\t';
}

void  SubstituteFunc (FcPattern *pattern, gpointer data)
{
  FcPatternAddString(pattern, FC_STYLE, (FcChar8 *)data);
  qTraceLogger("SubstituteFunc:style='%s'",(FcChar8 *)data);
}


PangoFontMap* FontPlatformData::m_fontMap = 0;
GHashTable* FontPlatformData::m_hashTable = 0;

static boost::unordered::unordered_map< std::string , const char * > pango_facename_map;


FontPlatformData::FontPlatformData(const FontDescription& fontDescription, const AtomicString& familyName)
    : m_context(0)
    , m_font(0)
    , m_size(fontDescription.computedSize())
    , m_syntheticBold(false)
    , m_syntheticOblique(false)
//  , m_scaledFont(0)
{
    FontPlatformData::init();

    CString stored_family = familyName.string().utf8();
    char const* families[] = {
      stored_family.data(),
      NULL
    };

    switch (fontDescription.genericFamily()) {
    case FontDescription::SerifFamily:
        families[1] = "serif";
        break;
    case FontDescription::SansSerifFamily:
        families[1] = "sans";
        break;
    case FontDescription::MonospaceFamily:
        families[1] = "monospace";
        break;
    case FontDescription::NoFamily:
    case FontDescription::StandardFamily:
    default:
        families[1] = "sans";
        break;
    }


    m_fontMap = PANGO_MAP_INSTANCE;
    m_context = pango_font_map_create_context( m_fontMap );

    {
	FcPattern *fcPattern;
	FcPattern *fcPatternMatched;
	FcPattern *fcPatternFilter;
	FcBool	retBool;
	FcFontSet*	fcFontSet;
	FcResult	fcResult;
        qTraceLogger( "set family to fontconfig :%s" , families[0] );

	fcPattern = FcNameParse((FcChar8 *)families[0]);
	retBool = FcConfigSubstitute(NULL, fcPattern, FcMatchPattern);

	if(FcTrue == retBool) {
          FcDefaultSubstitute(fcPattern);
          fcFontSet = FcFontSetCreate();
          fcPatternMatched = FcFontMatch(NULL, fcPattern, &fcResult);

          FcFontSetAdd(fcFontSet, fcPatternMatched);
          FcPatternDestroy(fcPattern);

          fcPatternFilter = FcPatternFilter(fcPatternMatched, NULL);

          double size = 0.0;
          FcPatternGetDouble( fcPatternFilter , FC_SIZE, 0, &size);
          qTraceLogger( "fontconfig fcsize :%f" , size );

          int fc_width;
          FcPatternGetInteger( fcPatternFilter , FC_WIDTH, 0, &fc_width);
          qTraceLogger( "fontconfig fc_width :%d" , fc_width );

          double aspect = 0.0;
          FcPatternGetDouble( fcPatternFilter , FC_ASPECT, 0, &aspect);
          qTraceLogger( "fontconfig fcaspect :%f" , aspect );

          double fc_pixelsize;
          FcPatternGetDouble( fcPatternFilter , FC_PIXEL_SIZE, 0, &fc_pixelsize);
          qTraceLogger( "fontconfig fc_pixelsize :%f" , fc_pixelsize );

          int fc_spacing;
          FcPatternGetInteger( fcPatternFilter , FC_SPACING, 0, &fc_spacing);
          qTraceLogger( "fontconfig fc_spacing :%d" , fc_spacing );

                
          FcChar8 *fcFamily = FcPatternFormat(fcPatternFilter, (FcChar8 *)"%{family}");
          FcChar8 *fcStyle  = FcPatternFormat(fcPatternFilter, (FcChar8 *)"%{style}");
          qTraceLogger( "fcfamily :%s" , fcFamily );
          qTraceLogger( "fcstyle  :%s" , fcStyle );

          Vector<String> matchFamilies;
          String::fromUTF8((const char*)fcFamily ).split( ',' , matchFamilies );

          Vector<String> matchStyles;
          String::fromUTF8((const char*)fcStyle ).split( ',' , matchStyles );

          FcStrFree(fcFamily);
          FcStrFree(fcStyle);

          FcPatternDestroy(fcPatternFilter);
          FcFontSetDestroy(fcFontSet);

          String originFamily = String::fromUTF8(familyName.string().utf8().data()).lower().removeCharacters(isWhitespace);

          //
          // let exact match family be first candidate in the list from fontconfig
          //
          size_t exact_match_pos = matchFamilies.find( familyName.string() );
          if( exact_match_pos != notFound ){
            String exact_match_family = matchFamilies.at( exact_match_pos );
            String exact_match_style  = matchStyles.at( exact_match_pos );
            matchFamilies.remove( exact_match_pos );
            matchStyles.remove( exact_match_pos );

            matchFamilies.insert( 0 , exact_match_family );
            matchStyles.insert( 0 , exact_match_style );
          }

          PangoFontFamily** pango_families = 0;
          int n_families = 0;
          if( pango_families == 0 ){
            pango_font_map_list_families(m_fontMap, &pango_families, &n_families );
          }

          //
          // show family list from pango
          //
          for( unsigned int i = 0; m_font == NULL && i < matchFamilies.size(); i++ ){
            for (int family = 0; m_font == NULL && family < n_families; family++){
              const char * familyname = pango_font_family_get_name(pango_families[family]);
              if( strcmp( familyname , matchFamilies[i].utf8().data() ) == 0 ){
                int n_faces;
                PangoFontFace **faces = 0;
                pango_font_family_list_faces( pango_families[ family ], &faces , &n_faces);
                for( int face = 0; m_font == NULL && face < n_faces; face++ ){
                  const char *facename = pango_font_face_get_face_name(faces[face]);
                  if( strcmp( facename , matchStyles[i].utf8().data() ) == 0 ){
                    std::string family_facename = std::string( familyname ) + ":" + facename;
   
                    if( pango_facename_map.find( family_facename ) == pango_facename_map.end() )
                      {
                        unsigned int strsize = strlen( facename ) + 1;
                        char *fn = ( char * ) malloc( strsize );
                        strcpy( fn , facename );
                        pango_facename_map[ family_facename ] = fn;
                      }

                    PangoFontDescription* description = pango_font_face_describe( faces[face] );
                    if (fontDescription.weight() >= FontWeight600){
                      pango_font_description_set_weight(description, PANGO_WEIGHT_BOLD);
                    }
                    if (fontDescription.italic()){
                      pango_font_description_set_style(description, PANGO_STYLE_ITALIC);
                    }
                    pango_font_description_set_absolute_size(description, fontDescription.computedSize() * PANGO_SCALE);

                    std::string request_decription_str = pango_font_description_to_string(description);
                    qTraceLogger( "request description:%s" , request_decription_str.c_str() );

                    pango_ft2_font_map_set_default_substitute ((PangoFT2FontMap *)m_fontMap, SubstituteFunc, ( gpointer ) pango_facename_map[ family_facename ] , NULL);
                    pango_context_set_font_description(m_context, description);
                    m_font = pango_font_map_load_font(m_fontMap, m_context, description);
                    pango_font_description_free(description);

                    PangoFontDescription* thisDesc = pango_font_describe(m_font);
                    std::string font_decription_str = pango_font_description_to_string(thisDesc);
                    qTraceLogger( "   font description:%s" , font_decription_str.c_str() );
                    pango_font_description_free(thisDesc);

                  }
                }
                g_free( faces );
              }

            }
          }
          g_free(pango_families);

	}
    }

    if( m_font == NULL ){

      for (unsigned int i = 0; !m_font && i < G_N_ELEMENTS(families); i++) {
        PangoFontDescription* description = pango_font_description_new();
        pango_font_description_set_family( description , families[i]);
        // FIXME: Map all FontWeight values to Pango font weights.
        if (fontDescription.weight() >= FontWeight600){
          pango_font_description_set_weight(description, PANGO_WEIGHT_BOLD);
        }
        if (fontDescription.italic()){
          pango_font_description_set_style(description, PANGO_STYLE_ITALIC);
        }
        pango_font_description_set_absolute_size(description, fontDescription.computedSize() * PANGO_SCALE);

        std::string request_decription_str = pango_font_description_to_string(description);
        qTraceLogger( "request description:%s" , request_decription_str.c_str() );

        pango_context_set_font_description(m_context, description);
        m_font = pango_font_map_load_font(m_fontMap, m_context, description);
        pango_font_description_free(description);

        PangoFontDescription* thisDesc = pango_font_describe(m_font);
        std::string font_decription_str = pango_font_description_to_string(thisDesc);
        qTraceLogger( "   font description:%s" , font_decription_str.c_str() );
        pango_font_description_free(thisDesc);
      }
    }
}

FontPlatformData::FontPlatformData(float size, bool bold, bool italic)
    : m_context(0)
    , m_font(0)
    , m_size(size)
    , m_syntheticBold(bold)
    , m_syntheticOblique(italic)
//  , m_scaledFont(0)
{
}

#if 0
FontPlatformData::FontPlatformData(cairo_font_face_t* fontFace, float size, bool bold, bool italic)
    : m_context(0)
    , m_font(0)
    , m_size(size)
    , m_syntheticBold(bold)
    , m_syntheticOblique(italic)
    , m_scaledFont(0)
{
    cairo_matrix_t fontMatrix;
    cairo_matrix_init_scale(&fontMatrix, size, size);
    cairo_matrix_t ctm;
    cairo_matrix_init_identity(&ctm);
    cairo_font_options_t* options = cairo_font_options_create();

    // We force antialiasing and disable hinting to provide consistent
    // typographic qualities for custom fonts on all platforms.
    cairo_font_options_set_hint_style(options, CAIRO_HINT_STYLE_NONE);
    cairo_font_options_set_antialias(options, CAIRO_ANTIALIAS_GRAY);

    m_scaledFont = cairo_scaled_font_create(fontFace, &fontMatrix, &ctm, options);
    cairo_font_options_destroy(options);
}
#endif

bool FontPlatformData::init()
{
    static bool initialized = false;
    if (initialized)
        return true;
    initialized = true;

    m_fontMap = PANGO_MAP_INSTANCE;

    if (!m_hashTable) {
        PangoFontFamily** families = 0;
        int n_families = 0;

        m_hashTable = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_object_unref);

        pango_font_map_list_families(m_fontMap, &families, &n_families);

        for (int family = 0; family < n_families; family++)
                g_hash_table_insert(m_hashTable,
                                    g_strdup(pango_font_family_get_name(families[family])),
                                    g_object_ref(families[family]));

        g_free(families);
    }

    return true;
}

FontPlatformData::~FontPlatformData()
{
    if (m_font && m_font != reinterpret_cast<PangoFont*>(-1)) {
        g_object_unref(m_font);
        m_font = 0;
    }

    if (m_context) {
        g_object_unref(m_context);
        m_context = 0;
    }
#if 0
    if (m_scaledFont) {
        cairo_scaled_font_destroy(m_scaledFont);
        m_scaledFont = 0;
    }
#endif
}

bool FontPlatformData::isFixedPitch()
{
    PangoFontDescription* description = pango_font_describe_with_absolute_size(m_font);
    PangoFontFamily* family = reinterpret_cast<PangoFontFamily*>(g_hash_table_lookup(m_hashTable, pango_font_description_get_family(description)));
    pango_font_description_free(description);
    if(family == NULL)
        return false;
    return pango_font_family_is_monospace(family);
}

FontPlatformData& FontPlatformData::operator=(const FontPlatformData& other)
{
    // Check for self-assignment.
    if (this == &other)
        return *this;

    m_size = other.m_size;
    m_syntheticBold = other.m_syntheticBold;
    m_syntheticOblique = other.m_syntheticOblique;
#if 0
    if (other.m_scaledFont)
        cairo_scaled_font_reference(other.m_scaledFont);
    if (m_scaledFont)
        cairo_scaled_font_destroy(m_scaledFont);
    m_scaledFont = other.m_scaledFont;
#endif
    if (other.m_font)
        g_object_ref(other.m_font);
    if (m_font)
        g_object_unref(m_font);
    m_font = other.m_font;

    if (other.m_context)
        g_object_ref(other.m_context);
    if (m_context)
        g_object_unref(m_context);
    m_context = other.m_context;

    return *this;
}

FontPlatformData::FontPlatformData(const FontPlatformData& other)
    : m_context(0)
    , m_font(0)
//  , m_scaledFont(0)
{
    *this = other;
}

bool FontPlatformData::operator==(const FontPlatformData& other) const
{
    if (m_font == other.m_font)
        return true;
    if (m_font == 0 || m_font == reinterpret_cast<PangoFont*>(-1)
        || other.m_font == 0 || other.m_font == reinterpret_cast<PangoFont*>(-1))
        return false;
    
    return false;
}

#ifndef NDEBUG
String FontPlatformData::description() const
{
    return String();
}
#endif

}
